// --------------------------------------------------------------------------------------------
//	FinCmdMain.cs
//	Subsembly FinTS API
//	Copyright  2004-2007 Subsembly GmbH
// --------------------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Specialized;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Reflection;

using Subsembly.FinTS;
using Subsembly.Swift;

namespace FinCmd
{
	/// <summary>
	/// 
	/// </summary>

	class FinCmdMain
	{
		/// <summary>
		/// The main entry point for the application.
		/// </summary>

		static int Main(string[] vsArgs)
		{
			// Mindestens ein Argument muss vorhanden sein und die Anzahl der Argumente muss
			// ungerade sein.

			if ((vsArgs.Length == 0) || ((vsArgs.Length & 0x0001) == 1))
			{
				_Usage();
				return -1;
			}

			// Die gewnschte Aktion und die zugehrigen Parameter werden aus der Kommandozeile
			// extrahiert.

			
			StringDictionary vsArgsDict = new StringDictionary();

			for (int i = 0; i < vsArgs.Length; i += 2)
			{
				string sArgName = vsArgs[i];
				string sArgValue = vsArgs[i + 1];

				vsArgsDict.Add(sArgName, sArgValue);
			}

			// Aktion ausfhren.

			int nResult = -1;

			FinCmdMain aMain = new FinCmdMain();

			try
			{
				nResult = aMain.DoAction(vsArgsDict);
			}
			catch (Exception x)
			{
				Console.Error.WriteLine("Exception " + x.Message + "!");
				Console.Error.WriteLine(x.ToString());
				Debug.Fail(x.Message);
			}
#if DEBUG
			//Console.ReadLine();
#endif
			return nResult;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="sAction"></param>
		/// <param name="vsArgsDict"></param>
		/// <returns></returns>

		internal int DoAction(StringDictionary vsArgsDict)
		{
			// In jedem Fall wird die PIN oder der Dialogkontext zur Fortfhrung bentigt.

			string sPIN = vsArgsDict["-pin"];
			string sResume = vsArgsDict["-resume"];

			if ((sPIN == null) && (sResume == null))
			{
				Console.Error.WriteLine("PIN fehlt!");
				return -1;
			}

	

			// Optional kann eine Datei fr den HBCI-Trace angegeben werden.

			string sTraceFile = vsArgsDict["-trace"];

			// Wird der Schalter -suspend angegeben, so wird nach der Aktion keine
			// Dialogbeendigung durchgefhrt, sondern einfach der Zustand in die angegebene
			// Datei gespeichert.

			string sSuspend = vsArgsDict["-suspend"];

			// Die IFinCmd Implementierung fr die gewnschte Aktion erstellen.

            IFinCmd aCmd = new FinCmdStatement();
			if (aCmd == null)
			{
				_Usage();
				Console.Error.WriteLine("Aktion nicht bekannt!");
				return -1;
			}

			//

			if (!aCmd.Parse(vsArgsDict))
			{
				return -1;
			}

			// Zuerst ermitteln wir den Bankkontakt aus den bergebenen Parametern. Danach
			// erstellen wir damit einen FinService.

			FinContact aContact = null;
			FinDialog aDialog = null;
			FinService aService = null;

			if (sResume != null)
			{
				aDialog = new FinDialog();
				aDialog.Load(sResume);
				aContact = aDialog.Contact;
			}
			else
			{
				aContact = _GetContact(vsArgsDict);
				if (aContact == null)
				{
					return -1;
				}
			}

			aService = _GetService(aContact, aDialog, vsArgsDict);
			if (aService == null)
			{
				return -1;
			}

			//

			int nResult = -2;

			try
			{
				aService.ClearDocket();

				// Mit dem FinService und dem IFinCmd die gewnschte Aktion durchfhren.

				if (!aService.Online)
				{
					if (!aService.LogOn(sPIN))
					{
						nResult = -3;
					}
				}

				//

				if (aService.Online)
				{
					nResult = aCmd.Execute(aService );

					if (sSuspend != null)
					{
						aService.Dialog.SaveAs(sSuspend);
					}
					else
					{
						aService.LogOff();
					}
				}

				// Falls der Bankkontakt aus einer Datei geladen wurde, so muss diese nun noch
				// gespeichert werden, damit auch alle am Bankkontakt erfolgten nderungen
				// erhalten bleiben.

				string sContactFile = vsArgsDict["-contactfile"];
				if (sContactFile != null)
				{
					aContact.SaveAs(sContactFile);
				}

				// Wurde eine Tracedatei angegeben, so wird der komplette HBCI Trace in diese
				// Datei geschrieben.

				if (sTraceFile != null)
				{
					StreamWriter sw = File.CreateText(sTraceFile);
					sw.Write(aService.Trace);
					sw.Close();
				}

				// Auftrag ausgefhrt. Zuerst geben wir den gesammelten Laufzettel aus,
				// danach die Antwortdaten, sofern welche vorhanden sind. Der Laufzettel wird
				// auf den Error-Kanal ausgegeben, damit er von der Antwortdaten leichter
				// getrennt werden kann.

				Console.Error.WriteLine(aService.Docket);

				if (nResult != -3)
				{
					string sResponseData = aCmd.GetResponseData(aService);
					if (sResponseData != null)
					{
						Console.WriteLine(sResponseData);
					}
				}
			}
			catch (Exception x)
			{
				Console.Error.WriteLine(aService.Trace);
				Console.Error.WriteLine(x.ToString());
			}
			finally
			{
				aService.Dispose();
				aService = null;
			}

			return nResult;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="vsArgsDict"></param>
		/// <returns></returns>

		static FinContact _GetContact(StringDictionary vsArgsDict)
		{
			FinContact aContact = null;

			string sContactFile = vsArgsDict["-contactfile"];
			string sContactName = vsArgsDict["-contactname"];

			if (sContactFile != null)
			{
				if (!File.Exists(sContactFile))
				{
					Console.WriteLine("Datei {0} nicht gefunden!", sContactFile);
					return null;
				}
				aContact = new FinContact();
				aContact.Load(sContactFile);
			}
			else if (sContactName != null)
			{
				aContact = FinContactFolder.Default.FindContact(sContactName);
				if (aContact == null)
				{
					Console.WriteLine("Bankkontakt {0} nicht gefunden!", sContactName);
					return null;
				}
			}
			else
			{
				string sCommAddress = vsArgsDict["-commaddress"];
				string sFinTSVersion = vsArgsDict["-fintsversion"];
				string sTanProcedure = vsArgsDict["-tanprocedure"];
				string sBankCode = vsArgsDict["-bankcode"];
				string sUserID = vsArgsDict["-userid"];
				string sCustID = vsArgsDict["-custid"];

				if ((sCommAddress == null) || (sBankCode == null) || (sUserID == null))
				{
					Console.WriteLine("Bankkontakt nicht vollstndig spezifiziert!");
					return null;
				}

				int nFinTSVersion = (sFinTSVersion != null) ? Int32.Parse(sFinTSVersion) : 300;

				aContact = new FinContact(sCommAddress, nFinTSVersion, sTanProcedure,
					sBankCode, sUserID);

				if (sCustID != null)
				{
					aContact["DefaultCustID"] = sCustID;
				}
			}

			return aContact;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="aContact"></param>
		/// <param name="aDialog"></param>
		/// <param name="vsArgsDict"></param>
		/// <returns></returns>

		static FinService _GetService(
			FinContact aContact,
			FinDialog aDialog,
			StringDictionary vsArgsDict)
		{
			Debug.Assert(aContact != null);

			// Als nchstes ermitteln wir das Konto fr die gewnschte Bankverbindung.

			string sAcctBankCode = vsArgsDict["-acctbankcode"];
			string sAcctNo = vsArgsDict["-acctno"];
			string sAcctCurrency = vsArgsDict["-acctcurrency"];

			if (sAcctBankCode == null)
			{
				sAcctBankCode = aContact.BankCode;
			}
			if (sAcctBankCode == null)
			{
				Console.WriteLine("Bankleitzahl zu Konto fehlt!");
				return null;
			}
			if ((sAcctBankCode.Length != 8) || !FinUtil.IsDigits(sAcctBankCode))
			{
				Console.WriteLine("Bankleitzahl zu Konto ist ungltig!");
				return null;
			}

			if (sAcctNo == null)
			{
				Console.WriteLine("Kontonummer fehlt!");
				return null;
			}
			if ((sAcctNo == "") || (sAcctNo.Length > 30))
			{
				Console.WriteLine("Kontonummer ist ungltig!");
				return null;
			}

			if (sAcctCurrency == null)
			{
				sAcctCurrency = "EUR";
			}
			else if ((sAcctCurrency.Length != 3) || !FinUtil.IsUpperAscii(sAcctCurrency))
			{
				Console.WriteLine("Kontowhrung ist ungltig!");
				return null;
			}

			// Die Bankverbindung ist jetzt vollstndig spezifiziert und wir knnen ein
			// FinService Objekt dafr anlegen.

			if (aDialog != null)
			{
				return new FinService(aDialog, sAcctBankCode, sAcctNo, sAcctCurrency);
			}
			else
			{
				return new FinService(aContact, sAcctBankCode, sAcctNo, sAcctCurrency);
			}
		}

	
		/// <summary>
		/// 
		/// </summary>

		static void _Usage()
		{
			Assembly aThisAssembly = Assembly.GetExecutingAssembly();
			Stream aStream = aThisAssembly.GetManifestResourceStream("Subsembly.FinCmdUsage.txt");
			StreamReader aReader = new StreamReader(aStream, Encoding.UTF8);
			string sUsage = aReader.ReadToEnd();
			aReader.Close();
			aStream.Close();

			Console.Error.WriteLine(sUsage);
		}
	}
}
